/**
 * AUTOKLASEN CHAT WIDGET
 * Version: 1.0.0
 * 
 * Integrare: Adaugă acest script pe orice pagină a site-ului autoklasen.ro
 * <script src="https://chat.autoklasen.ro/widget/autoklasen-chat.js" defer></script>
 */

(function() {
    'use strict';

    // ==========================================================================
    // CONFIGURARE
    // ==========================================================================
    const CONFIG = {
        apiBaseUrl: 'https://chat.autoklasen.ro/api/endpoints',
        triggerDelay: 10000,        // 10 secunde până la trigger automat
        cookieName: 'ak_visitor_id',
        cookieDays: 365,            // Cookie valid 1 an
        sseReconnectDelay: 3000,    // Reconectare SSE după 3 secunde
        maxReconnectAttempts: 5,
        debug: false                // Set true pentru logging
    };

    // ==========================================================================
    // UTILITĂȚI
    // ==========================================================================
    const Utils = {
        // Generare UUID v4
        generateUUID() {
            return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
                const r = Math.random() * 16 | 0;
                const v = c === 'x' ? r : (r & 0x3 | 0x8);
                return v.toString(16);
            });
        },

        // Cookie management
        setCookie(name, value, days) {
            const date = new Date();
            date.setTime(date.getTime() + (days * 24 * 60 * 60 * 1000));
            const expires = 'expires=' + date.toUTCString();
            document.cookie = name + '=' + value + ';' + expires + ';path=/;SameSite=Lax';
        },

        getCookie(name) {
            const nameEQ = name + '=';
            const ca = document.cookie.split(';');
            for (let i = 0; i < ca.length; i++) {
                let c = ca[i].trim();
                if (c.indexOf(nameEQ) === 0) {
                    return c.substring(nameEQ.length);
                }
            }
            return null;
        },

        // Format timestamp
        formatTime(date) {
            return date.toLocaleTimeString('ro-RO', { 
                hour: '2-digit', 
                minute: '2-digit' 
            });
        },

        // Escape HTML pentru securitate
        escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        },

        // Debug logging
        log(...args) {
            if (CONFIG.debug) {
                console.log('[AK Chat]', ...args);
            }
        },

        // AJAX helper
        async ajax(endpoint, data = {}, method = 'POST') {
            const url = CONFIG.apiBaseUrl + endpoint;
            
            try {
                const response = await fetch(url, {
                    method: method,
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: method !== 'GET' ? JSON.stringify(data) : undefined
                });

                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}`);
                }

                return await response.json();
            } catch (error) {
                Utils.log('AJAX Error:', error);
                throw error;
            }
        }
    };

    // ==========================================================================
    // STATE MANAGEMENT
    // ==========================================================================
    const State = {
        visitorId: null,
        sessionId: null,
        conversationId: null,
        isOpen: false,
        isMinimized: true,
        triggerShown: false,
        triggerDismissed: false,
        hasActiveConversation: false,
        unreadCount: 0,
        eventSource: null,
        reconnectAttempts: 0,
        pageEnterTime: Date.now(),
        currentPage: window.location.href,
        isFirstMessage: true
    };

    // ==========================================================================
    // UI COMPONENTS
    // ==========================================================================
    const UI = {
        container: null,
        button: null,
        window: null,
        messagesContainer: null,
        input: null,
        sendButton: null,
        statusElement: null,

        // Injectează CSS
        injectStyles() {
            if (document.getElementById('ak-chat-styles')) return;

            const link = document.createElement('link');
            link.id = 'ak-chat-styles';
            link.rel = 'stylesheet';
            link.href = CONFIG.apiBaseUrl.replace('/api/endpoints', '') + '/widget/autoklasen-chat.css';
            document.head.appendChild(link);
        },

        // Construiește HTML-ul widget-ului
        buildWidget() {
            const container = document.createElement('div');
            container.id = 'ak-chat-widget';
            container.innerHTML = `
                <!-- Buton Floating -->
                <button class="ak-chat-button" aria-label="Deschide chat">
                    <svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                        <path d="M20 2H4c-1.1 0-2 .9-2 2v18l4-4h14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zm0 14H5.17L4 17.17V4h16v12z"/>
                        <path d="M7 9h10v2H7zm0-3h10v2H7z"/>
                    </svg>
                </button>

                <!-- Fereastra Chat -->
                <div class="ak-chat-window">
                    <!-- Header -->
                    <div class="ak-chat-header">
                        <div class="ak-chat-avatar">P</div>
                        <div class="ak-chat-info">
                            <div class="ak-chat-name">Paul - Autoklasen</div>
                            <div class="ak-chat-status">
                                <span class="ak-status-text">Online</span>
                                <span class="ak-typing-indicator">
                                    <span></span><span></span><span></span>
                                </span>
                            </div>
                        </div>
                        <button class="ak-chat-minimize" aria-label="Minimizează">
                            <svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                <path d="M19 13H5v-2h14v2z"/>
                            </svg>
                        </button>
                    </div>

                    <!-- Mesaje -->
                    <div class="ak-chat-messages"></div>

                    <!-- Input -->
                    <div class="ak-chat-input-area">
                        <div class="ak-chat-input-wrapper">
                            <textarea 
                                class="ak-chat-input" 
                                placeholder="Scrieți un mesaj..." 
                                rows="1"
                                aria-label="Mesajul dumneavoastră"
                            ></textarea>
                        </div>
                        <button class="ak-chat-send" aria-label="Trimite" disabled>
                            <svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                <path d="M2.01 21L23 12 2.01 3 2 10l15 2-15 2z"/>
                            </svg>
                        </button>
                    </div>

                    <!-- Footer -->
                    <div class="ak-chat-footer">
                        Powered by <a href="https://autoklasen.ro" target="_blank">Autoklasen</a>
                    </div>
                </div>
            `;

            document.body.appendChild(container);

            // Salvează referințele
            this.container = container;
            this.button = container.querySelector('.ak-chat-button');
            this.window = container.querySelector('.ak-chat-window');
            this.messagesContainer = container.querySelector('.ak-chat-messages');
            this.input = container.querySelector('.ak-chat-input');
            this.sendButton = container.querySelector('.ak-chat-send');
            this.statusElement = container.querySelector('.ak-chat-status');
            this.minimizeButton = container.querySelector('.ak-chat-minimize');
        },

        // Bind event listeners
        bindEvents() {
            // Click pe buton - deschide/închide
            this.button.addEventListener('click', () => {
                if (State.isOpen) {
                    this.closeWindow();
                } else {
                    this.openWindow();
                }
            });

            // Click pe minimize
            this.minimizeButton.addEventListener('click', () => {
                this.closeWindow();
            });

            // Input - activează/dezactivează butonul send
            this.input.addEventListener('input', () => {
                this.sendButton.disabled = !this.input.value.trim();
                this.autoResizeInput();
            });

            // Enter pentru trimitere (Shift+Enter = linie nouă)
            this.input.addEventListener('keydown', (e) => {
                if (e.key === 'Enter' && !e.shiftKey) {
                    e.preventDefault();
                    if (this.input.value.trim()) {
                        Chat.sendMessage();
                    }
                }
            });

            // Click pe send
            this.sendButton.addEventListener('click', () => {
                if (this.input.value.trim()) {
                    Chat.sendMessage();
                }
            });

            // Track page visibility
            document.addEventListener('visibilitychange', () => {
                if (document.visibilityState === 'visible' && State.conversationId) {
                    // Reconectare SSE dacă s-a pierdut
                    if (!State.eventSource || State.eventSource.readyState === EventSource.CLOSED) {
                        Chat.startSSE();
                    }
                }
            });
        },

        // Auto-resize textarea
        autoResizeInput() {
            this.input.style.height = 'auto';
            this.input.style.height = Math.min(this.input.scrollHeight, 100) + 'px';
        },

        // Deschide fereastra
        openWindow() {
            State.isOpen = true;
            State.isMinimized = false;
            this.window.classList.add('ak-open');
            this.button.classList.remove('ak-attention', 'ak-has-unread');
            State.unreadCount = 0;
            
            // Focus pe input
            setTimeout(() => this.input.focus(), 300);

            // Marchează trigger ca afișat (dacă s-a deschis de la trigger)
            if (!State.triggerShown) {
                State.triggerShown = true;
                Tracking.logEvent('chat_opened', { trigger: 'manual' });
            }

            Utils.log('Window opened');
        },

        // Închide fereastra
        closeWindow() {
            State.isOpen = false;
            State.isMinimized = true;
            this.window.classList.remove('ak-open');
            this.button.classList.remove('ak-attention');
            
            // Dacă a fost trigger și l-a închis, marchează dismissed
            if (State.triggerShown && !State.hasActiveConversation) {
                State.triggerDismissed = true;
                Tracking.logEvent('trigger_dismissed');
            }

            Utils.log('Window closed');
        },

        // Adaugă mesaj în UI
        addMessage(message) {
            const messageEl = document.createElement('div');
            messageEl.className = `ak-message ak-message-${message.sender_type}`;

            let senderLabel = '';
            if (message.sender_type === 'operator' && message.sender_name) {
                senderLabel = `<div class="ak-message-sender">${Utils.escapeHtml(message.sender_name)}</div>`;
            }

            messageEl.innerHTML = `
                ${senderLabel}
                <div class="ak-message-content">${Utils.escapeHtml(message.content)}</div>
                <div class="ak-message-time">${Utils.formatTime(new Date(message.sent_at || Date.now()))}</div>
            `;

            this.messagesContainer.appendChild(messageEl);
            this.scrollToBottom();

            // Dacă fereastra e minimizată și e mesaj nou (nu de la vizitator)
            if (State.isMinimized && message.sender_type !== 'visitor') {
                State.unreadCount++;
                this.button.classList.add('ak-has-unread');
            }
        },

        // Adaugă mesaj de bun venit
        addWelcomeMessage(text) {
            const messageEl = document.createElement('div');
            messageEl.className = 'ak-message ak-message-welcome';
            messageEl.innerHTML = `
                <div class="ak-message-content">${Utils.escapeHtml(text)}</div>
            `;
            this.messagesContainer.appendChild(messageEl);
            this.scrollToBottom();
        },

        // Scroll la ultimul mesaj
        scrollToBottom() {
            this.messagesContainer.scrollTop = this.messagesContainer.scrollHeight;
        },

        // Arată indicator de typing
        showTyping() {
            this.statusElement.classList.add('ak-typing');
        },

        // Ascunde indicator de typing
        hideTyping() {
            this.statusElement.classList.remove('ak-typing');
        },

        // Actualizează status (Online/Offline/Operator name)
        updateStatus(text, isOperator = false) {
            const statusText = this.statusElement.querySelector('.ak-status-text');
            statusText.textContent = text;

            // Schimbă avatar dacă e operator
            const avatar = this.container.querySelector('.ak-chat-avatar');
            if (isOperator) {
                avatar.textContent = text.charAt(0).toUpperCase();
            } else {
                avatar.textContent = 'P';
            }
        },

        // Dezactivează input (conversație închisă)
        disableInput() {
            const inputArea = this.container.querySelector('.ak-chat-input-area');
            inputArea.classList.add('ak-disabled');
            this.input.disabled = true;
            this.sendButton.disabled = true;
        },

        // Trigger automat - atrage atenția
        showTrigger() {
            if (State.triggerDismissed || State.triggerShown) return;
            
            State.triggerShown = true;
            this.button.classList.add('ak-attention');
            
            // Deschide automat fereastra și arată mesaj
            this.openWindow();
            this.addWelcomeMessage('👋 Bună! Vă putem ajuta cu informații despre constatări sau reparații?');
            
            Tracking.logEvent('trigger_shown');
            Utils.log('Trigger shown');
        },

        // Curăță mesajele
        clearMessages() {
            this.messagesContainer.innerHTML = '';
        }
    };

    // ==========================================================================
    // TRACKING
    // ==========================================================================
    const Tracking = {
        // Identifică vizitatorul
        async identify() {
            // Verifică/generează visitor_id
            let visitorId = Utils.getCookie(CONFIG.cookieName);
            if (!visitorId) {
                visitorId = Utils.generateUUID();
                Utils.setCookie(CONFIG.cookieName, visitorId, CONFIG.cookieDays);
                Utils.log('New visitor:', visitorId);
            } else {
                Utils.log('Returning visitor:', visitorId);
            }

            State.visitorId = visitorId;

            // Trimite la server
            try {
                const response = await Utils.ajax('/visitor/identify.php', {
                    visitor_id: visitorId,
                    page_url: window.location.href,
                    page_title: document.title,
                    referrer: document.referrer,
                    user_agent: navigator.userAgent,
                    screen_width: window.screen.width,
                    screen_height: window.screen.height
                });

                if (response.success) {
                    State.sessionId = response.session_id;
                    
                    // Verifică dacă are conversație activă
                    if (response.active_conversation_id) {
                        State.conversationId = response.active_conversation_id;
                        State.hasActiveConversation = true;
                        State.isFirstMessage = false;
                        await Chat.loadHistory();
                        Chat.startSSE();
                        UI.button.classList.add('ak-has-unread');
                    }
                    
                    Utils.log('Identified:', response);
                }
            } catch (error) {
                Utils.log('Identify error:', error);
            }
        },

        // Track pageview
        async trackPageview() {
            if (!State.sessionId) return;

            const timeOnPrevPage = Date.now() - State.pageEnterTime;
            State.pageEnterTime = Date.now();

            try {
                await Utils.ajax('/visitor/pageview.php', {
                    visitor_id: State.visitorId,
                    session_id: State.sessionId,
                    page_url: window.location.href,
                    page_title: document.title,
                    time_on_previous_page: Math.round(timeOnPrevPage / 1000)
                });

                State.currentPage = window.location.href;
                Utils.log('Pageview tracked');
            } catch (error) {
                Utils.log('Pageview error:', error);
            }
        },

        // Log event generic
        async logEvent(eventType, eventData = {}) {
            if (!State.visitorId) return;

            try {
                await Utils.ajax('/visitor/pageview.php', {
                    visitor_id: State.visitorId,
                    session_id: State.sessionId,
                    event_type: eventType,
                    event_data: eventData
                });
            } catch (error) {
                Utils.log('Event log error:', error);
            }
        },

        // Pornește timer pentru trigger
        startTriggerTimer() {
            setTimeout(() => {
                if (!State.triggerDismissed && !State.hasActiveConversation && !State.isOpen) {
                    UI.showTrigger();
                }
            }, CONFIG.triggerDelay);
        }
    };

    // ==========================================================================
    // CHAT LOGIC
    // ==========================================================================
    const Chat = {
        // Pornește conversație nouă
        async startConversation() {
            try {
                const response = await Utils.ajax('/chat/start.php', {
                    visitor_id: State.visitorId,
                    session_id: State.sessionId,
                    page_url: window.location.href,
                    trigger_type: State.triggerShown ? 'auto' : 'manual'
                });

                if (response.success) {
                    State.conversationId = response.conversation_id;
                    State.hasActiveConversation = true;
                    this.startSSE();
                    Utils.log('Conversation started:', response.conversation_id);
                    return true;
                }
            } catch (error) {
                Utils.log('Start conversation error:', error);
            }
            return false;
        },

        // Trimite mesaj
        async sendMessage() {
            const content = UI.input.value.trim();
            if (!content) return;

            // Dacă nu există conversație, pornește una
            if (!State.conversationId) {
                const started = await this.startConversation();
                if (!started) {
                    UI.addMessage({
                        sender_type: 'ai',
                        content: 'Ne pare rău, a apărut o eroare. Vă rugăm să sunați la 0371 719 191.',
                        sent_at: new Date()
                    });
                    return;
                }
            }

            // Curăță input
            UI.input.value = '';
            UI.sendButton.disabled = true;
            UI.autoResizeInput();

            // Adaugă mesajul vizitatorului în UI imediat
            UI.addMessage({
                sender_type: 'visitor',
                content: content,
                sent_at: new Date()
            });

            // Arată typing
            UI.showTyping();

            // Trimite la server
            try {
                const response = await Utils.ajax('/chat/message.php', {
                    visitor_id: State.visitorId,
                    conversation_id: State.conversationId,
                    content: content,
                    page_url: window.location.href,
                    is_first_message: State.isFirstMessage
                });

                State.isFirstMessage = false;
                UI.hideTyping();

                if (response.success && response.ai_response) {
                    // Răspunsul AI vine prin SSE, dar ca backup îl adăugăm și aici
                    // doar dacă nu a venit deja prin SSE (verificăm un flag)
                    if (!response.via_sse) {
                        UI.addMessage({
                            sender_type: 'ai',
                            content: response.ai_response,
                            sent_at: new Date()
                        });
                    }
                }
            } catch (error) {
                UI.hideTyping();
                Utils.log('Send message error:', error);
                
                // Fallback message
                UI.addMessage({
                    sender_type: 'ai',
                    content: 'Ne pare rău, a apărut o eroare. Vă rugăm să sunați la 0371 719 191.',
                    sent_at: new Date()
                });
            }
        },

        // Încarcă istoricul conversației
        async loadHistory() {
            if (!State.conversationId) return;

            try {
                const response = await Utils.ajax('/chat/history.php', {
                    conversation_id: State.conversationId,
                    visitor_id: State.visitorId
                });

                if (response.success && response.messages) {
                    UI.clearMessages();
                    response.messages.forEach(msg => {
                        UI.addMessage(msg);
                    });

                    // Actualizează status dacă e operator activ
                    if (response.status === 'operator_active' && response.operator_name) {
                        UI.updateStatus(response.operator_name, true);
                    }
                }
            } catch (error) {
                Utils.log('Load history error:', error);
            }
        },

        // Pornește SSE pentru mesaje real-time
        startSSE() {
            if (!State.conversationId) return;

            // Închide conexiunea existentă dacă există
            if (State.eventSource) {
                State.eventSource.close();
            }

            const url = `${CONFIG.apiBaseUrl}/chat/stream.php?conversation_id=${State.conversationId}&visitor_id=${State.visitorId}`;
            
            State.eventSource = new EventSource(url);

            State.eventSource.onopen = () => {
                Utils.log('SSE connected');
                State.reconnectAttempts = 0;
            };

            State.eventSource.onmessage = (event) => {
                try {
                    const data = JSON.parse(event.data);
                    
                    if (data.type === 'message' && data.sender_type !== 'visitor') {
                        UI.addMessage({
                            sender_type: data.sender_type,
                            content: data.content,
                            sender_name: data.sender_name,
                            sent_at: data.sent_at
                        });

                        // Play notification sound dacă e minimizat
                        if (State.isMinimized) {
                            this.playNotificationSound();
                        }
                    }

                    if (data.type === 'operator_joined') {
                        UI.updateStatus(data.operator_name, true);
                        UI.addMessage({
                            sender_type: 'system',
                            content: `${data.operator_name} s-a alăturat conversației.`,
                            sent_at: new Date()
                        });
                    }

                    if (data.type === 'typing') {
                        if (data.is_typing) {
                            UI.showTyping();
                        } else {
                            UI.hideTyping();
                        }
                    }

                    if (data.type === 'closed') {
                        UI.disableInput();
                        State.hasActiveConversation = false;
                    }
                } catch (error) {
                    Utils.log('SSE parse error:', error);
                }
            };

            State.eventSource.onerror = () => {
                Utils.log('SSE error, attempting reconnect...');
                State.eventSource.close();

                // Reconectare cu backoff
                if (State.reconnectAttempts < CONFIG.maxReconnectAttempts) {
                    State.reconnectAttempts++;
                    setTimeout(() => {
                        if (State.hasActiveConversation) {
                            this.startSSE();
                        }
                    }, CONFIG.sseReconnectDelay * State.reconnectAttempts);
                }
            };
        },

        // Play notification sound
        playNotificationSound() {
            try {
                const audio = new Audio(CONFIG.apiBaseUrl.replace('/api/endpoints', '') + '/widget/assets/notification.mp3');
                audio.volume = 0.5;
                audio.play().catch(() => {}); // Ignoră erori dacă browser blochează autoplay
            } catch (error) {
                // Ignoră
            }
        },

        // Închide conversația
        async closeConversation() {
            if (!State.conversationId) return;

            try {
                await Utils.ajax('/chat/close.php', {
                    conversation_id: State.conversationId,
                    visitor_id: State.visitorId
                });
            } catch (error) {
                Utils.log('Close conversation error:', error);
            }

            if (State.eventSource) {
                State.eventSource.close();
            }

            State.conversationId = null;
            State.hasActiveConversation = false;
        }
    };

    // ==========================================================================
    // INIȚIALIZARE
    // ==========================================================================
    async function init() {
        Utils.log('Initializing...');

        // Injectează CSS
        UI.injectStyles();

        // Construiește widget-ul
        UI.buildWidget();

        // Bind events
        UI.bindEvents();

        // Identifică vizitatorul
        await Tracking.identify();

        // Pornește timer pentru trigger automat
        Tracking.startTriggerTimer();

        // Track navigare (SPA support)
        let lastUrl = window.location.href;
        const observer = new MutationObserver(() => {
            if (window.location.href !== lastUrl) {
                lastUrl = window.location.href;
                Tracking.trackPageview();
            }
        });
        observer.observe(document.body, { childList: true, subtree: true });

        // Track la beforeunload
        window.addEventListener('beforeunload', () => {
            Tracking.trackPageview();
        });

        Utils.log('Initialized successfully');
    }

    // Pornește când DOM-ul e gata
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

    // Expune API public pentru debugging
    window.AutoklasenChat = {
        open: () => UI.openWindow(),
        close: () => UI.closeWindow(),
        getState: () => ({ ...State }),
        setDebug: (val) => { CONFIG.debug = val; }
    };

})();
