/**
 * Autoklasen Chat - Service Worker
 * Enables PWA installation and basic caching
 */

const CACHE_NAME = 'ak-chat-v1';
const ASSETS_TO_CACHE = [
    '/operator/',
    '/operator/index.html',
    '/operator/css/styles.css',
    '/operator/js/app.js',
    '/operator/manifest.json',
    '/operator/assets/icon-192.png',
    '/operator/assets/icon-512.png',
    '/operator/assets/sounds/notification.mp3'
];

// Install - cache assets
self.addEventListener('install', (event) => {
    event.waitUntil(
        caches.open(CACHE_NAME)
            .then((cache) => {
                console.log('Caching assets...');
                return cache.addAll(ASSETS_TO_CACHE);
            })
            .then(() => self.skipWaiting())
    );
});

// Activate - cleanup old caches
self.addEventListener('activate', (event) => {
    event.waitUntil(
        caches.keys()
            .then((cacheNames) => {
                return Promise.all(
                    cacheNames
                        .filter((name) => name !== CACHE_NAME)
                        .map((name) => caches.delete(name))
                );
            })
            .then(() => self.clients.claim())
    );
});

// Fetch - network first, fallback to cache
self.addEventListener('fetch', (event) => {
    // Skip API calls and SSE
    if (event.request.url.includes('/api/') || 
        event.request.url.includes('stream.php')) {
        return;
    }

    event.respondWith(
        fetch(event.request)
            .then((response) => {
                // Clone and cache successful responses
                if (response.status === 200) {
                    const responseClone = response.clone();
                    caches.open(CACHE_NAME)
                        .then((cache) => cache.put(event.request, responseClone));
                }
                return response;
            })
            .catch(() => {
                // Fallback to cache
                return caches.match(event.request)
                    .then((response) => {
                        if (response) {
                            return response;
                        }
                        // Return offline page for navigation
                        if (event.request.mode === 'navigate') {
                            return caches.match('/operator/index.html');
                        }
                        return new Response('Offline', { status: 503 });
                    });
            })
    );
});

// Push notifications (pentru viitor)
self.addEventListener('push', (event) => {
    if (!event.data) return;

    const data = event.data.json();
    
    event.waitUntil(
        self.registration.showNotification(data.title || 'Autoklasen Chat', {
            body: data.body || 'Mesaj nou',
            icon: '/operator/assets/icon-192.png',
            badge: '/operator/assets/icon-192.png',
            vibrate: [200, 100, 200],
            tag: 'chat-notification',
            renotify: true,
            data: data.data || {}
        })
    );
});

// Notification click
self.addEventListener('notificationclick', (event) => {
    event.notification.close();

    event.waitUntil(
        clients.matchAll({ type: 'window', includeUncontrolled: true })
            .then((clientList) => {
                // Focus existing window
                for (const client of clientList) {
                    if (client.url.includes('/operator/') && 'focus' in client) {
                        return client.focus();
                    }
                }
                // Open new window
                if (clients.openWindow) {
                    return clients.openWindow('/operator/');
                }
            })
    );
});
