/**
 * Autoklasen Chat - Operator Dashboard
 * Main Application
 */

(function() {
    'use strict';

    // ==========================================================================
    // CONFIG
    // ==========================================================================
    const CONFIG = {
        apiBaseUrl: '/api/endpoints',
        sseReconnectDelay: 3000,
        maxReconnectAttempts: 10
    };

    // ==========================================================================
    // STATE
    // ==========================================================================
    const State = {
        operator: null,
        token: null,
        conversations: [],
        currentConversation: null,
        eventSource: null,
        reconnectAttempts: 0,
        soundEnabled: true
    };

    // ==========================================================================
    // DOM ELEMENTS
    // ==========================================================================
    const DOM = {
        // Screens
        loginScreen: document.getElementById('login-screen'),
        dashboardScreen: document.getElementById('dashboard-screen'),
        
        // Login
        loginForm: document.getElementById('login-form'),
        emailInput: document.getElementById('email'),
        passwordInput: document.getElementById('password'),
        loginError: document.getElementById('login-error'),
        
        // Header
        operatorName: document.getElementById('operator-name'),
        connectionStatus: document.getElementById('connection-status'),
        logoutBtn: document.getElementById('logout-btn'),
        
        // Conversations
        conversationsList: document.getElementById('conversations-list'),
        refreshBtn: document.getElementById('refresh-btn'),
        
        // Chat
        chatPlaceholder: document.getElementById('chat-placeholder'),
        chatContainer: document.getElementById('chat-container'),
        chatVisitorId: document.getElementById('chat-visitor-id'),
        chatStatus: document.getElementById('chat-status'),
        chatMessages: document.getElementById('chat-messages'),
        messageInput: document.getElementById('message-input'),
        sendBtn: document.getElementById('send-btn'),
        joinBtn: document.getElementById('join-btn'),
        closeChatBtn: document.getElementById('close-chat-btn'),
        backBtn: document.getElementById('back-btn'),
        
        // Visitor Info
        visitorInfo: document.getElementById('visitor-info'),
        infoVisits: document.getElementById('info-visits'),
        infoCity: document.getElementById('info-city'),
        infoPage: document.getElementById('info-page'),
        infoReferrer: document.getElementById('info-referrer'),
        
        // Other
        mainContent: document.querySelector('.main-content'),
        notificationSound: document.getElementById('notification-sound'),
        toastContainer: document.getElementById('toast-container')
    };

    // ==========================================================================
    // UTILITIES
    // ==========================================================================
    const Utils = {
        async ajax(endpoint, data = null, method = 'POST') {
            const url = CONFIG.apiBaseUrl + endpoint;
            const options = {
                method,
                headers: { 'Content-Type': 'application/json' },
                credentials: 'include'
            };
            
            if (data && method !== 'GET') {
                options.body = JSON.stringify(data);
            }
            
            const response = await fetch(url, options);
            return response.json();
        },

        formatTime(dateStr) {
            const date = new Date(dateStr);
            return date.toLocaleTimeString('ro-RO', { hour: '2-digit', minute: '2-digit' });
        },

        formatRelativeTime(dateStr) {
            const date = new Date(dateStr);
            const now = new Date();
            const diff = Math.floor((now - date) / 1000);
            
            if (diff < 60) return 'acum';
            if (diff < 3600) return Math.floor(diff / 60) + ' min';
            if (diff < 86400) return Math.floor(diff / 3600) + ' ore';
            return date.toLocaleDateString('ro-RO');
        },

        escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        },

        truncate(str, len) {
            if (!str) return '';
            return str.length > len ? str.substring(0, len) + '...' : str;
        }
    };

    // ==========================================================================
    // TOAST NOTIFICATIONS
    // ==========================================================================
    const Toast = {
        show(message, type = 'info', duration = 4000) {
            const toast = document.createElement('div');
            toast.className = `toast ${type}`;
            
            const icons = { success: '✓', error: '✕', info: 'ℹ' };
            toast.innerHTML = `
                <span class="toast-icon">${icons[type] || icons.info}</span>
                <span class="toast-message">${Utils.escapeHtml(message)}</span>
            `;
            
            DOM.toastContainer.appendChild(toast);
            
            setTimeout(() => {
                toast.classList.add('toast-out');
                setTimeout(() => toast.remove(), 300);
            }, duration);
        },

        success(msg) { this.show(msg, 'success'); },
        error(msg) { this.show(msg, 'error'); },
        info(msg) { this.show(msg, 'info'); }
    };

    // ==========================================================================
    // AUTH
    // ==========================================================================
    const Auth = {
        async checkSession() {
            try {
                const result = await Utils.ajax('/operator/auth.php', { action: 'verify' });
                if (result.success) {
                    State.operator = result.operator;
                    State.token = result.token;
                    this.showDashboard();
                    return true;
                }
            } catch (e) {
                console.error('Session check failed:', e);
            }
            return false;
        },

        async login(email, password) {
            const result = await Utils.ajax('/operator/auth.php', {
                action: 'login',
                email,
                password
            });
            
            if (result.success) {
                State.operator = result.operator;
                State.token = result.token;
                this.showDashboard();
                return true;
            }
            
            throw new Error(result.error || 'Eroare la autentificare');
        },

        async logout() {
            try {
                await Utils.ajax('/operator/auth.php', { action: 'logout' });
            } catch (e) {
                console.error('Logout error:', e);
            }
            
            State.operator = null;
            State.token = null;
            
            if (State.eventSource) {
                State.eventSource.close();
                State.eventSource = null;
            }
            
            this.showLogin();
        },

        showLogin() {
            DOM.loginScreen.classList.add('active');
            DOM.dashboardScreen.classList.remove('active');
            DOM.loginError.textContent = '';
            DOM.loginForm.reset();
        },

        showDashboard() {
            DOM.loginScreen.classList.remove('active');
            DOM.dashboardScreen.classList.add('active');
            DOM.operatorName.textContent = State.operator.name;
            
            // Start SSE și încarcă conversațiile
            SSE.connect();
            Conversations.load();
        }
    };

    // ==========================================================================
    // SSE (Server-Sent Events)
    // ==========================================================================
    const SSE = {
        connect() {
            if (State.eventSource) {
                State.eventSource.close();
            }

            this.updateStatus('connecting');

            const url = `${CONFIG.apiBaseUrl}/operator/stream.php?operator_id=${State.operator.id}`;
            State.eventSource = new EventSource(url, { withCredentials: true });

            State.eventSource.addEventListener('connected', () => {
                console.log('SSE connected');
                State.reconnectAttempts = 0;
                this.updateStatus('connected');
            });

            State.eventSource.addEventListener('new_conversation', (e) => {
                const conv = JSON.parse(e.data);
                console.log('New conversation:', conv);
                Conversations.addNew(conv);
                Notifications.play();
                Toast.info('Conversație nouă!');
            });

            State.eventSource.addEventListener('message', (e) => {
                const msg = JSON.parse(e.data);
                console.log('New message:', msg);
                
                // Actualizează lista de conversații
                Conversations.updatePreview(msg.conversation_id, msg.content, msg.sender_type);
                
                // Dacă e conversația curentă, adaugă mesajul
                if (State.currentConversation && State.currentConversation.id == msg.conversation_id) {
                    Chat.addMessage(msg);
                } else {
                    Notifications.play();
                }
            });

            State.eventSource.addEventListener('operator_joined', (e) => {
                const data = JSON.parse(e.data);
                console.log('Operator joined:', data);
                Conversations.updateStatus(data.conversation_id, 'operator_active', data.operator_name);
            });

            State.eventSource.addEventListener('conversation_closed', (e) => {
                const data = JSON.parse(e.data);
                console.log('Conversation closed:', data);
                Conversations.remove(data.conversation_id);
            });

            State.eventSource.addEventListener('reconnect', () => {
                console.log('SSE reconnect requested');
                this.reconnect();
            });

            State.eventSource.onerror = () => {
                console.log('SSE error');
                this.updateStatus('disconnected');
                this.reconnect();
            };
        },

        reconnect() {
            if (State.eventSource) {
                State.eventSource.close();
                State.eventSource = null;
            }

            if (State.reconnectAttempts < CONFIG.maxReconnectAttempts) {
                State.reconnectAttempts++;
                const delay = CONFIG.sseReconnectDelay * State.reconnectAttempts;
                console.log(`Reconnecting in ${delay}ms...`);
                setTimeout(() => this.connect(), delay);
            } else {
                Toast.error('Conexiune pierdută. Reîncarcă pagina.');
            }
        },

        updateStatus(status) {
            DOM.connectionStatus.className = 'status-badge status-' + status;
            const texts = {
                connecting: 'Conectare...',
                connected: 'Online',
                disconnected: 'Deconectat'
            };
            DOM.connectionStatus.querySelector('.status-text').textContent = texts[status] || status;
        }
    };

    // ==========================================================================
    // CONVERSATIONS
    // ==========================================================================
    const Conversations = {
        async load() {
            try {
                const result = await Utils.ajax('/operator/conversations.php', null, 'GET');
                if (result.success) {
                    State.conversations = result.conversations;
                    this.render();
                }
            } catch (e) {
                console.error('Load conversations error:', e);
                Toast.error('Eroare la încărcarea conversațiilor');
            }
        },

        render() {
            if (State.conversations.length === 0) {
                DOM.conversationsList.innerHTML = `
                    <div class="empty-state">
                        <span class="empty-icon">💬</span>
                        <p>Nicio conversație activă</p>
                    </div>
                `;
                return;
            }

            DOM.conversationsList.innerHTML = State.conversations.map(conv => {
                const isActive = State.currentConversation && State.currentConversation.id === conv.id;
                const isNew = conv.status === 'ai_active';
                const statusClass = conv.status === 'ai_active' ? 'ai' : 'operator';
                const statusText = conv.status === 'ai_active' ? 'AI activ' : conv.operator_name || 'Operator';
                
                return `
                    <div class="conversation-item ${isActive ? 'active' : ''}" data-id="${conv.id}">
                        <div class="conv-header">
                            <span class="conv-visitor">
                                Vizitator #${conv.id}
                                ${isNew ? '<span class="conv-badge">NOU</span>' : ''}
                            </span>
                            <span class="conv-time">${Utils.formatRelativeTime(conv.last_message_time || conv.started_at)}</span>
                        </div>
                        <div class="conv-preview">${Utils.escapeHtml(conv.last_message_preview || 'Fără mesaje')}</div>
                        <div class="conv-status ${statusClass}">${statusText}</div>
                    </div>
                `;
            }).join('');

            // Bind click events
            DOM.conversationsList.querySelectorAll('.conversation-item').forEach(item => {
                item.addEventListener('click', () => {
                    const id = parseInt(item.dataset.id);
                    this.select(id);
                });
            });
        },

        addNew(conv) {
            // Verifică dacă există deja
            if (State.conversations.find(c => c.id === conv.id)) return;
            
            State.conversations.unshift(conv);
            this.render();
        },

        updatePreview(conversationId, content, senderType) {
            const conv = State.conversations.find(c => c.id == conversationId);
            if (conv) {
                conv.last_message_preview = Utils.truncate(content, 80);
                conv.last_message_time = new Date().toISOString();
                this.render();
            }
        },

        updateStatus(conversationId, status, operatorName) {
            const conv = State.conversations.find(c => c.id == conversationId);
            if (conv) {
                conv.status = status;
                conv.operator_name = operatorName;
                this.render();
            }
        },

        remove(conversationId) {
            State.conversations = State.conversations.filter(c => c.id != conversationId);
            
            if (State.currentConversation && State.currentConversation.id == conversationId) {
                Chat.close();
            }
            
            this.render();
        },

        select(id) {
            const conv = State.conversations.find(c => c.id === id);
            if (conv) {
                State.currentConversation = conv;
                Chat.open(conv);
                this.render();
                
                // Mobile: show chat
                DOM.mainContent.classList.add('chat-open');
            }
        }
    };

    // ==========================================================================
    // CHAT
    // ==========================================================================
    const Chat = {
        async open(conversation) {
            DOM.chatPlaceholder.style.display = 'none';
            DOM.chatContainer.style.display = 'flex';
            
            DOM.chatVisitorId.textContent = `Vizitator #${conversation.id}`;
            DOM.chatStatus.textContent = conversation.status === 'ai_active' ? 'AI activ' : 'Operator activ';
            
            // Update buttons
            const isMyConversation = conversation.operator_id == State.operator.id;
            const isClosed = conversation.status === 'closed';
            
            DOM.joinBtn.style.display = conversation.status === 'ai_active' ? '' : 'none';
            DOM.closeChatBtn.style.display = isClosed ? 'none' : '';
            DOM.messageInput.disabled = conversation.status === 'ai_active' || isClosed;
            DOM.sendBtn.disabled = conversation.status === 'ai_active' || isClosed;
            
            // Încarcă detaliile
            try {
                const result = await Utils.ajax('/operator/join.php', {
                    conversation_id: conversation.id
                });
                
                if (result.success) {
                    // Update visitor info
                    if (result.visitor) {
                        DOM.infoVisits.textContent = result.visitor.total_visits || '-';
                        DOM.infoCity.textContent = result.visitor.city || '-';
                        DOM.infoReferrer.textContent = Utils.truncate(result.visitor.referrer, 50) || '-';
                    }
                    
                    if (result.page_views && result.page_views.length > 0) {
                        DOM.infoPage.textContent = Utils.truncate(result.page_views[0].page_url, 50);
                    }
                    
                    // Render messages
                    DOM.chatMessages.innerHTML = '';
                    result.messages.forEach(msg => this.addMessage(msg));
                    this.scrollToBottom();
                    
                    // Dacă a preluat conversația, actualizează UI
                    if (conversation.status === 'ai_active') {
                        conversation.status = 'operator_active';
                        conversation.operator_id = State.operator.id;
                        DOM.chatStatus.textContent = 'Operator activ';
                        DOM.joinBtn.style.display = 'none';
                        DOM.messageInput.disabled = false;
                        DOM.sendBtn.disabled = false;
                        DOM.messageInput.focus();
                        Conversations.render();
                    }
                }
            } catch (e) {
                console.error('Open chat error:', e);
                Toast.error('Eroare la încărcarea conversației');
            }
        },

        close() {
            State.currentConversation = null;
            DOM.chatPlaceholder.style.display = 'flex';
            DOM.chatContainer.style.display = 'none';
            DOM.mainContent.classList.remove('chat-open');
            Conversations.render();
        },

        addMessage(msg) {
            const messageEl = document.createElement('div');
            messageEl.className = `message ${msg.sender_type}`;
            
            let senderLabel = '';
            if (msg.sender_type === 'operator') {
                senderLabel = `<div class="message-sender">${Utils.escapeHtml(msg.sender_name || State.operator.name)}</div>`;
            }
            
            messageEl.innerHTML = `
                ${senderLabel}
                <div class="message-content">${Utils.escapeHtml(msg.content)}</div>
                <div class="message-time">${Utils.formatTime(msg.sent_at || new Date())}</div>
            `;
            
            DOM.chatMessages.appendChild(messageEl);
            this.scrollToBottom();
        },

        async send() {
            const content = DOM.messageInput.value.trim();
            if (!content || !State.currentConversation) return;
            
            DOM.messageInput.value = '';
            DOM.sendBtn.disabled = true;
            
            // Adaugă mesajul imediat în UI
            this.addMessage({
                sender_type: 'operator',
                content: content,
                sent_at: new Date().toISOString()
            });
            
            try {
                const result = await Utils.ajax('/operator/message.php', {
                    conversation_id: State.currentConversation.id,
                    content: content
                });
                
                if (!result.success) {
                    Toast.error(result.error || 'Eroare la trimitere');
                }
            } catch (e) {
                console.error('Send message error:', e);
                Toast.error('Eroare la trimiterea mesajului');
            }
            
            DOM.sendBtn.disabled = false;
            DOM.messageInput.focus();
        },

        async closeConversation() {
            if (!State.currentConversation) return;
            
            if (!confirm('Sigur vrei să închizi această conversație?')) return;
            
            try {
                const result = await Utils.ajax('/operator/close.php', {
                    conversation_id: State.currentConversation.id
                });
                
                if (result.success) {
                    Toast.success('Conversație închisă');
                    Conversations.remove(State.currentConversation.id);
                    this.close();
                } else {
                    Toast.error(result.error || 'Eroare');
                }
            } catch (e) {
                console.error('Close conversation error:', e);
                Toast.error('Eroare la închiderea conversației');
            }
        },

        scrollToBottom() {
            DOM.chatMessages.scrollTop = DOM.chatMessages.scrollHeight;
        }
    };

    // ==========================================================================
    // NOTIFICATIONS
    // ==========================================================================
    const Notifications = {
        play() {
            if (!State.soundEnabled) return;
            
            try {
                DOM.notificationSound.currentTime = 0;
                DOM.notificationSound.play().catch(() => {});
            } catch (e) {
                // Ignore
            }
        }
    };

    // ==========================================================================
    // EVENT BINDINGS
    // ==========================================================================
    function bindEvents() {
        // Login form
        DOM.loginForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const btn = DOM.loginForm.querySelector('button[type="submit"]');
            btn.classList.add('loading');
            DOM.loginError.textContent = '';
            
            try {
                await Auth.login(DOM.emailInput.value, DOM.passwordInput.value);
            } catch (err) {
                DOM.loginError.textContent = err.message;
            }
            
            btn.classList.remove('loading');
        });

        // Logout
        DOM.logoutBtn.addEventListener('click', () => Auth.logout());

        // Refresh conversations
        DOM.refreshBtn.addEventListener('click', () => Conversations.load());

        // Send message
        DOM.sendBtn.addEventListener('click', () => Chat.send());
        
        DOM.messageInput.addEventListener('keydown', (e) => {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                Chat.send();
            }
        });

        // Auto-resize textarea
        DOM.messageInput.addEventListener('input', () => {
            DOM.messageInput.style.height = 'auto';
            DOM.messageInput.style.height = Math.min(DOM.messageInput.scrollHeight, 120) + 'px';
        });

        // Join conversation
        DOM.joinBtn.addEventListener('click', async () => {
            if (State.currentConversation) {
                await Chat.open(State.currentConversation);
            }
        });

        // Close conversation
        DOM.closeChatBtn.addEventListener('click', () => Chat.closeConversation());

        // Back button (mobile)
        DOM.backBtn.addEventListener('click', () => Chat.close());

        // Visitor info toggle
        DOM.visitorInfo.querySelector('.visitor-info-toggle').addEventListener('click', () => {
            DOM.visitorInfo.classList.toggle('collapsed');
        });

        // Visibility change - reconnect SSE
        document.addEventListener('visibilitychange', () => {
            if (document.visibilityState === 'visible' && State.operator) {
                if (!State.eventSource || State.eventSource.readyState === EventSource.CLOSED) {
                    SSE.connect();
                }
            }
        });
    }

    // ==========================================================================
    // INIT
    // ==========================================================================
    async function init() {
        bindEvents();
        
        // Check existing session
        const hasSession = await Auth.checkSession();
        
        if (!hasSession) {
            Auth.showLogin();
        }
        
        // Register service worker
        if ('serviceWorker' in navigator) {
            try {
                await navigator.serviceWorker.register('sw.js');
                console.log('Service Worker registered');
            } catch (e) {
                console.log('Service Worker registration failed:', e);
            }
        }
    }

    // Start
    init();

})();
