<?php
/**
 * Identifică vizitatorul (nou sau existent)
 * POST /api/endpoints/visitor/identify.php
 * 
 * Input: { visitor_id?: string, landing_page: string, referrer?: string }
 * Output: { visitor_id: string, session_id: int, is_new: bool }
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: https://autoklasen.ro');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    die(json_encode(['error' => 'Method not allowed']));
}

require_once __DIR__ . '/../../config/database.php';

try {
    $input = json_decode(file_get_contents('php://input'), true);
    $db = getDB();
    
    $visitorId = $input['visitor_id'] ?? null;
    $landingPage = $input['landing_page'] ?? '/';
    $referrer = $input['referrer'] ?? null;
    $isNew = false;
    
    // Detectare browser și device din User-Agent
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    $browser = detectBrowser($userAgent);
    $device = detectDevice($userAgent);
    
    // UTM parameters
    $utmSource = $input['utm_source'] ?? null;
    $utmMedium = $input['utm_medium'] ?? null;
    $utmCampaign = $input['utm_campaign'] ?? null;
    
    $now = date('Y-m-d H:i:s');
    
    // Verifică dacă vizitatorul există
    if ($visitorId) {
        $stmt = $db->prepare("SELECT id FROM visitors WHERE id = ?");
        $stmt->execute([$visitorId]);
        $exists = $stmt->fetch();
        
        if ($exists) {
            // Actualizează last_seen și incrementează vizitele
            $stmt = $db->prepare("
                UPDATE visitors 
                SET last_seen = ?, total_visits = total_visits + 1 
                WHERE id = ?
            ");
            $stmt->execute([$now, $visitorId]);
        } else {
            $visitorId = null; // ID invalid, creăm unul nou
        }
    }
    
    // Creează vizitator nou dacă nu există
    if (!$visitorId) {
        $visitorId = generateUUID();
        $isNew = true;
        
        $stmt = $db->prepare("
            INSERT INTO visitors (id, first_seen, last_seen, browser, device) 
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([$visitorId, $now, $now, $browser, $device]);
    }
    
    // Creează sesiune nouă
    $stmt = $db->prepare("
        INSERT INTO sessions (visitor_id, started_at, landing_page, referrer, utm_source, utm_medium, utm_campaign) 
        VALUES (?, ?, ?, ?, ?, ?, ?)
    ");
    $stmt->execute([$visitorId, $now, $landingPage, $referrer, $utmSource, $utmMedium, $utmCampaign]);
    $sessionId = $db->lastInsertId();
    
    // Loghează evenimentul
    $stmt = $db->prepare("
        INSERT INTO events (visitor_id, session_id, event_type, event_data) 
        VALUES (?, ?, 'session_start', ?)
    ");
    $stmt->execute([$visitorId, $sessionId, json_encode([
        'landing_page' => $landingPage,
        'referrer' => $referrer,
        'is_new_visitor' => $isNew
    ])]);
    
    // Înregistrează prima pagină vizualizată
    $stmt = $db->prepare("
        INSERT INTO page_views (session_id, page_url, entered_at) 
        VALUES (?, ?, ?)
    ");
    $stmt->execute([$sessionId, $landingPage, $now]);
    
    echo json_encode([
        'success' => true,
        'visitor_id' => $visitorId,
        'session_id' => $sessionId,
        'is_new' => $isNew
    ]);

} catch (Exception $e) {
    error_log("Identify error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Server error']);
}

/**
 * Generează UUID v4
 */
function generateUUID() {
    $data = random_bytes(16);
    $data[6] = chr(ord($data[6]) & 0x0f | 0x40);
    $data[8] = chr(ord($data[8]) & 0x3f | 0x80);
    return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
}

/**
 * Detectează browserul din User-Agent
 */
function detectBrowser($ua) {
    if (preg_match('/Chrome\/[\d.]+/i', $ua) && !preg_match('/Edge|Edg/i', $ua)) return 'Chrome';
    if (preg_match('/Firefox\/[\d.]+/i', $ua)) return 'Firefox';
    if (preg_match('/Safari\/[\d.]+/i', $ua) && !preg_match('/Chrome/i', $ua)) return 'Safari';
    if (preg_match('/Edge|Edg/i', $ua)) return 'Edge';
    if (preg_match('/MSIE|Trident/i', $ua)) return 'IE';
    return 'Other';
}

/**
 * Detectează tipul de device
 */
function detectDevice($ua) {
    if (preg_match('/Mobile|Android.*Mobile|iPhone|iPod/i', $ua)) return 'mobile';
    if (preg_match('/iPad|Android(?!.*Mobile)|Tablet/i', $ua)) return 'tablet';
    return 'desktop';
}