<?php
/**
 * Autoklasen Chat - SSE Stream pentru Operator
 * GET /api/endpoints/operator/stream.php
 * 
 * Trimite:
 * - Conversații noi
 * - Mesaje noi în conversațiile active
 * - Vizitatori noi pe site
 */

header('Content-Type: text/event-stream');
header('Cache-Control: no-cache');
header('Connection: keep-alive');
header('Access-Control-Allow-Origin: *');
header('X-Accel-Buffering: no'); // Pentru nginx

// Disable output buffering
if (ob_get_level()) ob_end_clean();

require_once __DIR__ . '/../../config/database.php';

session_start();

// Verifică autentificare prin sesiune sau token
$operatorId = $_SESSION['operator_id'] ?? null;

if (!$operatorId && isset($_GET['token'])) {
    // Verifică token-ul dacă nu e sesiune (pentru SSE care nu trimite cookies)
    // În producție ar trebui verificat în DB
    $operatorId = $_GET['operator_id'] ?? null;
}

if (!$operatorId) {
    echo "event: error\n";
    echo "data: " . json_encode(['error' => 'Not authenticated']) . "\n\n";
    exit;
}

try {
    $pdo = getDBConnection();
    
    // Update status online
    $stmt = $pdo->prepare("UPDATE operators SET is_online = TRUE, last_active = NOW() WHERE id = ?");
    $stmt->execute([$operatorId]);
    
} catch (Exception $e) {
    echo "event: error\n";
    echo "data: " . json_encode(['error' => 'Database error']) . "\n\n";
    exit;
}

// Trimite ping inițial
echo "event: connected\n";
echo "data: " . json_encode(['status' => 'connected', 'operator_id' => $operatorId]) . "\n\n";
flush();

// Track ultima verificare
$lastMessageId = (int)($_GET['last_message_id'] ?? 0);
$lastConversationId = (int)($_GET['last_conversation_id'] ?? 0);
$lastEventId = (int)($_GET['last_event_id'] ?? 0);

// Obține ultimele ID-uri dacă nu sunt furnizate
if ($lastMessageId === 0) {
    $stmt = $pdo->query("SELECT MAX(id) FROM messages");
    $lastMessageId = (int)$stmt->fetchColumn() ?: 0;
}

if ($lastConversationId === 0) {
    $stmt = $pdo->query("SELECT MAX(id) FROM conversations");
    $lastConversationId = (int)$stmt->fetchColumn() ?: 0;
}

if ($lastEventId === 0) {
    $stmt = $pdo->query("SELECT MAX(id) FROM events");
    $lastEventId = (int)$stmt->fetchColumn() ?: 0;
}

$startTime = time();
$maxRuntime = 55; // Reconectare după 55 secunde

while (true) {
    // Verifică timeout
    if (time() - $startTime > $maxRuntime) {
        echo "event: reconnect\n";
        echo "data: " . json_encode(['reason' => 'timeout']) . "\n\n";
        break;
    }
    
    // Verifică dacă conexiunea e încă deschisă
    if (connection_aborted()) {
        break;
    }
    
    try {
        // 1. Verifică conversații noi
        $stmt = $pdo->prepare("
            SELECT 
                c.id, c.visitor_id, c.status, c.started_at, c.page_started,
                v.total_visits, v.city,
                (SELECT content FROM messages WHERE conversation_id = c.id ORDER BY sent_at DESC LIMIT 1) as last_message
            FROM conversations c
            LEFT JOIN visitors v ON c.visitor_id = v.id
            WHERE c.id > ? AND c.status = 'ai_active'
            ORDER BY c.id ASC
        ");
        $stmt->execute([$lastConversationId]);
        $newConversations = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($newConversations as $conv) {
            echo "event: new_conversation\n";
            echo "data: " . json_encode($conv) . "\n\n";
            $lastConversationId = max($lastConversationId, $conv['id']);
        }
        
        // 2. Verifică mesaje noi (din toate conversațiile active)
        $stmt = $pdo->prepare("
            SELECT 
                m.id, m.conversation_id, m.sender_type, m.content, m.sent_at,
                c.visitor_id
            FROM messages m
            JOIN conversations c ON m.conversation_id = c.id
            WHERE m.id > ? 
            AND c.status IN ('ai_active', 'operator_active')
            ORDER BY m.id ASC
        ");
        $stmt->execute([$lastMessageId]);
        $newMessages = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($newMessages as $msg) {
            echo "event: message\n";
            echo "data: " . json_encode($msg) . "\n\n";
            $lastMessageId = max($lastMessageId, $msg['id']);
        }
        
        // 3. Verifică evenimente (operator joined, etc.)
        $stmt = $pdo->prepare("
            SELECT id, event_type, event_data, created_at
            FROM events 
            WHERE id > ?
            ORDER BY id ASC
            LIMIT 20
        ");
        $stmt->execute([$lastEventId]);
        $newEvents = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($newEvents as $event) {
            echo "event: " . $event['event_type'] . "\n";
            echo "data: " . $event['event_data'] . "\n\n";
            $lastEventId = max($lastEventId, $event['id']);
        }
        
        // Flush dacă au fost date
        if (!empty($newConversations) || !empty($newMessages) || !empty($newEvents)) {
            flush();
        }
        
        // Update last_active periodic
        if (time() % 30 === 0) {
            $stmt = $pdo->prepare("UPDATE operators SET last_active = NOW() WHERE id = ?");
            $stmt->execute([$operatorId]);
        }
        
    } catch (Exception $e) {
        echo "event: error\n";
        echo "data: " . json_encode(['error' => 'Query error']) . "\n\n";
        flush();
    }
    
    // Ping keep-alive
    echo ": ping\n\n";
    flush();
    
    // Așteaptă 1 secundă
    sleep(1);
}

// Marchează offline la deconectare
try {
    $stmt = $pdo->prepare("UPDATE operators SET is_online = FALSE, last_active = NOW() WHERE id = ?");
    $stmt->execute([$operatorId]);
} catch (Exception $e) {
    // Ignoră erori la cleanup
}
