<?php
/**
 * Autoklasen Chat - Operator preia conversația
 * POST /api/endpoints/operator/join.php
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/../../config/database.php';

session_start();

// Verifică autentificare
if (!isset($_SESSION['operator_id'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Not authenticated']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);
$conversationId = $input['conversation_id'] ?? null;

if (!$conversationId) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'conversation_id required']);
    exit;
}

try {
    $pdo = getDBConnection();
    
    // Verifică conversația
    $stmt = $pdo->prepare("SELECT id, status, operator_id, visitor_id FROM conversations WHERE id = ?");
    $stmt->execute([$conversationId]);
    $conversation = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$conversation) {
        http_response_code(404);
        echo json_encode(['success' => false, 'error' => 'Conversation not found']);
        exit;
    }
    
    // Verifică dacă e deja preluată de altcineva
    if ($conversation['status'] === 'operator_active' && $conversation['operator_id'] != $_SESSION['operator_id']) {
        http_response_code(409);
        echo json_encode(['success' => false, 'error' => 'Conversație deja preluată de alt operator']);
        exit;
    }
    
    // Verifică dacă e închisă
    if ($conversation['status'] === 'closed') {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Conversația este închisă']);
        exit;
    }
    
    // Preia conversația
    $stmt = $pdo->prepare("
        UPDATE conversations 
        SET status = 'operator_active', 
            operator_id = ?
        WHERE id = ?
    ");
    $stmt->execute([$_SESSION['operator_id'], $conversationId]);
    
    // Log event pentru SSE
    $stmt = $pdo->prepare("
        INSERT INTO events (visitor_id, session_id, event_type, event_data) 
        VALUES (?, NULL, 'operator_joined', ?)
    ");
    $stmt->execute([
        $conversation['visitor_id'],
        json_encode([
            'conversation_id' => $conversationId,
            'operator_id' => $_SESSION['operator_id'],
            'operator_name' => $_SESSION['operator_name']
        ])
    ]);
    
    // Încarcă istoricul
    $stmt = $pdo->prepare("
        SELECT id, sender_type, content, sent_at 
        FROM messages 
        WHERE conversation_id = ? 
        ORDER BY sent_at ASC
    ");
    $stmt->execute([$conversationId]);
    $messages = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Info vizitator
    $stmt = $pdo->prepare("
        SELECT v.*, s.landing_page, s.referrer 
        FROM visitors v
        LEFT JOIN sessions s ON s.visitor_id = v.id
        WHERE v.id = ?
        ORDER BY s.started_at DESC
        LIMIT 1
    ");
    $stmt->execute([$conversation['visitor_id']]);
    $visitor = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Pagini vizitate în sesiunea curentă
    $stmt = $pdo->prepare("
        SELECT page_url, page_title, entered_at, time_on_page
        FROM page_views pv
        JOIN sessions s ON pv.session_id = s.id
        WHERE s.visitor_id = ?
        ORDER BY pv.entered_at DESC
        LIMIT 10
    ");
    $stmt->execute([$conversation['visitor_id']]);
    $pageViews = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'conversation_id' => $conversationId,
        'messages' => $messages,
        'visitor' => $visitor,
        'page_views' => $pageViews
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Server error']);
}
