<?php
/**
 * Autoklasen Chat - Operator Authentication
 * POST /api/endpoints/operator/auth.php
 * 
 * Actions: login, logout, verify
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/../../config/database.php';

session_start();

// Citește input
$input = json_decode(file_get_contents('php://input'), true);
$action = $input['action'] ?? 'verify';

try {
    $pdo = getDBConnection();
    
    switch ($action) {
        case 'login':
            handleLogin($pdo, $input);
            break;
            
        case 'logout':
            handleLogout($pdo);
            break;
            
        case 'verify':
            handleVerify($pdo);
            break;
            
        default:
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'Invalid action']);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Server error']);
}

/**
 * Login operator
 */
function handleLogin($pdo, $input) {
    $email = trim($input['email'] ?? '');
    $password = $input['password'] ?? '';
    
    if (empty($email) || empty($password)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'error' => 'Email și parolă obligatorii']);
        return;
    }
    
    // Caută operatorul
    $stmt = $pdo->prepare("SELECT id, name, email, password_hash, avatar_url FROM operators WHERE email = ?");
    $stmt->execute([$email]);
    $operator = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$operator || !password_verify($password, $operator['password_hash'])) {
        http_response_code(401);
        echo json_encode(['success' => false, 'error' => 'Email sau parolă incorectă']);
        return;
    }
    
    // Setează sesiune
    $_SESSION['operator_id'] = $operator['id'];
    $_SESSION['operator_name'] = $operator['name'];
    $_SESSION['operator_email'] = $operator['email'];
    
    // Update last_active și is_online
    $stmt = $pdo->prepare("UPDATE operators SET is_online = TRUE, last_active = NOW() WHERE id = ?");
    $stmt->execute([$operator['id']]);
    
    // Generează token simplu pentru SSE
    $token = bin2hex(random_bytes(32));
    $_SESSION['operator_token'] = $token;
    
    echo json_encode([
        'success' => true,
        'operator' => [
            'id' => $operator['id'],
            'name' => $operator['name'],
            'email' => $operator['email'],
            'avatar_url' => $operator['avatar_url']
        ],
        'token' => $token
    ]);
}

/**
 * Logout operator
 */
function handleLogout($pdo) {
    if (isset($_SESSION['operator_id'])) {
        // Update is_online
        $stmt = $pdo->prepare("UPDATE operators SET is_online = FALSE, last_active = NOW() WHERE id = ?");
        $stmt->execute([$_SESSION['operator_id']]);
    }
    
    // Distruge sesiunea
    session_destroy();
    
    echo json_encode(['success' => true]);
}

/**
 * Verifică dacă operatorul e autentificat
 */
function handleVerify($pdo) {
    if (!isset($_SESSION['operator_id'])) {
        http_response_code(401);
        echo json_encode(['success' => false, 'error' => 'Not authenticated']);
        return;
    }
    
    // Verifică dacă operatorul încă există
    $stmt = $pdo->prepare("SELECT id, name, email, avatar_url FROM operators WHERE id = ?");
    $stmt->execute([$_SESSION['operator_id']]);
    $operator = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$operator) {
        session_destroy();
        http_response_code(401);
        echo json_encode(['success' => false, 'error' => 'Operator not found']);
        return;
    }
    
    // Update last_active
    $stmt = $pdo->prepare("UPDATE operators SET last_active = NOW() WHERE id = ?");
    $stmt->execute([$operator['id']]);
    
    echo json_encode([
        'success' => true,
        'operator' => [
            'id' => $operator['id'],
            'name' => $operator['name'],
            'email' => $operator['email'],
            'avatar_url' => $operator['avatar_url']
        ],
        'token' => $_SESSION['operator_token'] ?? null
    ]);
}
