<?php
/**
 * SSE Stream pentru mesaje în timp real
 * GET /api/endpoints/chat/stream.php?conversation_id=X&last_id=Y
 * 
 * Trimite mesaje noi către client în timp real
 */

header('Content-Type: text/event-stream');
header('Cache-Control: no-cache');
header('Connection: keep-alive');
header('Access-Control-Allow-Origin: https://autoklasen.ro');
header('X-Accel-Buffering: no'); // Dezactivează buffering nginx (dacă e cazul)

// Previne timeout-ul PHP
set_time_limit(0);
ignore_user_abort(false);

require_once __DIR__ . '/../../config/database.php';

$conversationId = $_GET['conversation_id'] ?? null;
$lastId = $_GET['last_id'] ?? 0;

if (!$conversationId) {
    echo "event: error\n";
    echo "data: {\"error\": \"conversation_id required\"}\n\n";
    exit;
}

$db = getDB();

// Verifică dacă conversația există
$stmt = $db->prepare("SELECT id, status FROM conversations WHERE id = ?");
$stmt->execute([$conversationId]);
$conversation = $stmt->fetch();

if (!$conversation) {
    echo "event: error\n";
    echo "data: {\"error\": \"Conversation not found\"}\n\n";
    exit;
}

// Trimite un ping inițial
echo "event: connected\n";
echo "data: {\"status\": \"connected\", \"conversation_id\": $conversationId}\n\n";
ob_flush();
flush();

$retryCount = 0;
$maxRetries = 1800; // 30 minute (la 1 secundă interval)

while ($retryCount < $maxRetries) {
    // Verifică dacă clientul s-a deconectat
    if (connection_aborted()) {
        exit;
    }
    
    try {
        // Caută mesaje noi
        $stmt = $db->prepare("
            SELECT id, sender_type, sender_id, content, sent_at 
            FROM messages 
            WHERE conversation_id = ? AND id > ?
            ORDER BY id ASC
        ");
        $stmt->execute([$conversationId, $lastId]);
        $messages = $stmt->fetchAll();
        
        if ($messages) {
            foreach ($messages as $msg) {
                $data = json_encode([
                    'id' => $msg['id'],
                    'sender_type' => $msg['sender_type'],
                    'content' => $msg['content'],
                    'sent_at' => $msg['sent_at']
                ]);
                
                echo "event: message\n";
                echo "data: $data\n\n";
                
                $lastId = $msg['id'];
            }
            ob_flush();
            flush();
        }
        
        // Verifică dacă statusul conversației s-a schimbat
        $stmt = $db->prepare("SELECT status, operator_id FROM conversations WHERE id = ?");
        $stmt->execute([$conversationId]);
        $currentStatus = $stmt->fetch();
        
        if ($currentStatus['status'] !== $conversation['status']) {
            $statusData = json_encode([
                'status' => $currentStatus['status'],
                'operator_id' => $currentStatus['operator_id']
            ]);
            
            echo "event: status_change\n";
            echo "data: $statusData\n\n";
            ob_flush();
            flush();
            
            $conversation['status'] = $currentStatus['status'];
        }
        
        // Dacă conversația e închisă, trimite eveniment și închide conexiunea
        if ($currentStatus['status'] === 'closed') {
            echo "event: closed\n";
            echo "data: {\"message\": \"Conversation closed\"}\n\n";
            ob_flush();
            flush();
            exit;
        }
        
    } catch (Exception $e) {
        error_log("SSE error: " . $e->getMessage());
        
        echo "event: error\n";
        echo "data: {\"error\": \"Server error\"}\n\n";
        ob_flush();
        flush();
    }
    
    // Trimite ping la fiecare 15 secunde pentru a menține conexiunea
    if ($retryCount % 15 === 0) {
        echo "event: ping\n";
        echo "data: {\"time\": " . time() . "}\n\n";
        ob_flush();
        flush();
    }
    
    sleep(1);
    $retryCount++;
}

// Timeout - închide conexiunea
echo "event: timeout\n";
echo "data: {\"message\": \"Connection timeout, please reconnect\"}\n\n";
ob_flush();
flush();