<?php
/**
 * Trimite un mesaj și primește răspuns de la Paul (Claude)
 * POST /api/endpoints/chat/message.php
 * 
 * Input: { conversation_id: int, visitor_id: string, message: string, context?: object }
 * Output: { success: bool, visitor_message_id: int, ai_message_id: int, ai_response: string }
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: https://autoklasen.ro');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    die(json_encode(['error' => 'Method not allowed']));
}

require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../../config/claude.php';

try {
    $input = json_decode(file_get_contents('php://input'), true);
    $db = getDB();
    
    $conversationId = $input['conversation_id'] ?? null;
    $visitorId = $input['visitor_id'] ?? null;
    $message = trim($input['message'] ?? '');
    $context = $input['context'] ?? [];
    
    if (!$conversationId || !$visitorId || !$message) {
        http_response_code(400);
        die(json_encode(['error' => 'conversation_id, visitor_id and message required']));
    }
    
    // Verifică conversația și statusul
    $stmt = $db->prepare("
        SELECT id, status, visitor_id 
        FROM conversations 
        WHERE id = ? AND visitor_id = ?
    ");
    $stmt->execute([$conversationId, $visitorId]);
    $conversation = $stmt->fetch();
    
    if (!$conversation) {
        http_response_code(404);
        die(json_encode(['error' => 'Conversation not found']));
    }
    
    $now = date('Y-m-d H:i:s');
    
    // Salvează mesajul vizitatorului
    $stmt = $db->prepare("
        INSERT INTO messages (conversation_id, sender_type, sender_id, content, sent_at) 
        VALUES (?, 'visitor', ?, ?, ?)
    ");
    $stmt->execute([$conversationId, $visitorId, $message, $now]);
    $visitorMessageId = $db->lastInsertId();
    
    // Dacă operatorul a preluat conversația, nu mai răspunde AI-ul
    if ($conversation['status'] === 'operator_active') {
        echo json_encode([
            'success' => true,
            'visitor_message_id' => $visitorMessageId,
            'ai_message_id' => null,
            'ai_response' => null,
            'handled_by' => 'operator'
        ]);
        exit;
    }
    
    // Construiește istoricul conversației pentru Claude
    $stmt = $db->prepare("
        SELECT sender_type, content 
        FROM messages 
        WHERE conversation_id = ? 
        ORDER BY sent_at ASC
    ");
    $stmt->execute([$conversationId]);
    $history = $stmt->fetchAll();
    
    $claudeMessages = [];
    foreach ($history as $msg) {
        $role = ($msg['sender_type'] === 'visitor') ? 'user' : 'assistant';
        $claudeMessages[] = [
            'role' => $role,
            'content' => $msg['content']
        ];
    }
    
    // Obține paginile vizitate pentru context
    $stmt = $db->prepare("
        SELECT s.id as session_id 
        FROM conversations c 
        JOIN sessions s ON c.session_id = s.id 
        WHERE c.id = ?
    ");
    $stmt->execute([$conversationId]);
    $sessionData = $stmt->fetch();
    
    $pagesVisited = [];
    $currentPage = $context['current_page'] ?? '/';
    
    if ($sessionData) {
        $stmt = $db->prepare("
            SELECT page_url 
            FROM page_views 
            WHERE session_id = ? 
            ORDER BY entered_at ASC
        ");
        $stmt->execute([$sessionData['session_id']]);
        $pages = $stmt->fetchAll();
        foreach ($pages as $page) {
            $pagesVisited[] = $page['page_url'];
        }
    }
    
    // Determină dacă e primul mesaj (pentru salut)
    $isFirstMessage = count($claudeMessages) === 1;
    
    // Calculează timpul pe site
    $stmt = $db->prepare("
        SELECT TIMESTAMPDIFF(SECOND, started_at, NOW()) as time_on_site 
        FROM sessions 
        WHERE id = ?
    ");
    $stmt->execute([$sessionData['session_id']]);
    $timeData = $stmt->fetch();
    $timeOnSite = $timeData['time_on_site'] ?? 0;
    
    // Construiește contextul pentru Claude
    $claudeContext = [
        'current_page' => $currentPage,
        'pages_visited' => $pagesVisited,
        'time_on_site' => $timeOnSite,
        'is_first_message' => $isFirstMessage
    ];
    
    // Apelează Claude
    $aiResponse = callClaude($claudeMessages, $claudeContext);
    
    if ($aiResponse === null) {
        // Eroare la Claude - salvăm un mesaj de fallback
        $aiResponse = "Bună! Pentru această solicitare, vă rog să sunați la 0371 719 191.";
        error_log("Claude API failed for conversation $conversationId");
    }
    
    // Salvează răspunsul AI
    $stmt = $db->prepare("
        INSERT INTO messages (conversation_id, sender_type, content, sent_at) 
        VALUES (?, 'ai', ?, ?)
    ");
    $stmt->execute([$conversationId, $aiResponse, date('Y-m-d H:i:s')]);
    $aiMessageId = $db->lastInsertId();
    
    echo json_encode([
        'success' => true,
        'visitor_message_id' => $visitorMessageId,
        'ai_message_id' => $aiMessageId,
        'ai_response' => $aiResponse,
        'handled_by' => 'ai'
    ]);

} catch (Exception $e) {
    error_log("Chat message error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Server error']);
}